-- name: create-companies-table
--
-- The role_prefix is so we know which role to seach for with row
-- level security.
CREATE TABLE companies (
  name character varying(255) PRIMARY KEY CHECK (char_length(trim(name)) > 0)
);

-- name: ensure-company-name-unique
--
-- Create an index based on lowercase letters, to keep names unique.
CREATE UNIQUE INDEX companies_sanatise_name ON companies (
  REMOVE_WHITESPACE(LOWER(name))
);

-- name: company-members
--
-- Need a better name, but this holds the mapping of users that belong
-- to a company.
CREATE TABLE company_members (
  user_name character varying(40) REFERENCES users(user_name) ON UPDATE CASCADE,
  company_name character varying(255) REFERENCES companies(name) ON UPDATE CASCADE,
  is_admin boolean NOT NULL DEFAULT false,
  PRIMARY KEY (user_name,company_name)
);

-- name: enable-row-level-security
ALTER TABLE companies ENABLE ROW LEVEL SECURITY;

-- name: create-company-membership-query-function
--
-- Function that takes a username and a company name, and returns true
-- if that user is a member of that company.
CREATE OR REPLACE FUNCTION member_of_company(inUsername TEXT, inCompany TEXT)
RETURNS BOOLEAN AS
$BODY$
DECLARE
  result BOOLEAN = false;
BEGIN
  SELECT true
    INTO result
    FROM company_members
   WHERE $1 = user_name
     AND $2 = company_name;

  RETURN result;
END;
$BODY$
LANGUAGE 'plpgsql' IMMUTABLE
SECURITY DEFINER
SET search_path = public, pg_temp;

-- name: create-company-admin-query-function
--
-- Function that takes a username and a company name, and returns true
-- if that user is an admin of that company (membership to that
-- company is implied.)
CREATE OR REPLACE FUNCTION admin_of_company(inUsername TEXT, inCompany TEXT)
RETURNS BOOLEAN AS
$BODY$
DECLARE
  result BOOLEAN = false;
BEGIN
  SELECT true
    INTO result
    FROM company_members
   WHERE $1 = user_name
     AND $2 = company_name
     AND is_admin IS true;

  RETURN result;
END;
$BODY$
LANGUAGE 'plpgsql' IMMUTABLE
SECURITY DEFINER
SET search_path = public, pg_temp;

-- name: grant-select-public
GRANT SELECT ON companies TO PUBLIC;

-- name: grant-all-to-directgps-admin
--
-- Admins can do all.
GRANT SELECT,UPDATE,INSERT,DELETE ON companies TO directgps_admin;

CREATE POLICY company_admin_all ON companies TO directgps_admin
 USING (true) WITH CHECK (true);

-- name: create-companies-user-select-policy
--
-- Users can read their own data.
-- Admins can read everyone in their company, DirectGPS admins can do
-- all.
CREATE POLICY company_user_read_data ON companies
   FOR
SELECT
 USING (member_of_company(current_user,name) OR
        admin_of_company(current_user,name) OR
        is_admin(current_user));

-- name: create-companies-admin-user-select-policy
--
-- The admins of companies should be able to see their users details.
-- This will take a user_name and company_name and return a list of
-- users they can view.
CREATE OR REPLACE FUNCTION users_company_admin_can_view(inUsername TEXT)
RETURNS TABLE (
  user_name character varying(40)
) AS
$BODY$
BEGIN
  RETURN QUERY
    SELECT c2.user_name
               FROM company_members AS c1
    LEFT OUTER JOIN company_members AS c2
                 ON c1.company_name = c2.company_name
              WHERE c1.user_name = $1
                AND c1.is_admin IS true;
END;
$BODY$
LANGUAGE 'plpgsql' IMMUTABLE
SECURITY DEFINER
SET search_path = public, pg_temp;

-- name: change-user-select-policy
--
-- Change the users policy, so company admins can view users in their
-- company.
DROP POLICY user_read_own_data ON users;

-- name: create-company-users-select-policy
--
-- Admins of companies can view their users.
CREATE POLICY user_company_admin_and_own_read ON users
  FOR SELECT USING
    (user_name IN (SELECT users_company_admin_can_view(current_user))
     OR user_name = current_user
     OR (SELECT is_admin(current_user)));
